#include "SCI_Boot.h"

// Data section where SCIA control registers 
// reside
#pragma DATA_SECTION(SCIARegs,".SCIARegs");
volatile struct SCI_REGS SCIARegs;
                              

// Private functions
Uint16 SCIA_CheckKeyVal_app(void);
void SCIA_ReservedFn_app(void);
Uint32 SCIA_GetLongData_app(void);
Uint16 SCIA_GetWordData_app(void);
Uint16 SCIA_GetByteData_app(void);
Uint16 SCIA_FIFO_GetWordData(void);
Uint16 SCIA_FIFO_GetWordData_swapEndian(void);
void SCIA_Init(void);
void SCIA_AutobaudLock(void);
extern void SCI_SendStatus(char *);
void	SCI_SendHexWord(Uint16);
void	SCI_SendHexLong(Uint32);


//#################################################
// Uint16 SCIA_CheckKeyVal(void)
//-------------------------------------------------
// The header of the datafile should have a proper
// key value of 0x08 0xAA.  If it does not, then
// we either have a bad data file or we are not
// booting properly.  If this is the case, return
// an error to the main routine. 
//-------------------------------------------------
                              
Uint16 SCIA_CheckKeyVal_app()
{
    Uint16 wordData;
    
    wordData = SCIA_FIFO_GetWordData();
    if(wordData != EIGHT_BIT_HEADER) return ERROR;

    // No error found
    return NO_ERROR;
}

//#################################################
// void SCIA_ReservedFn(void)
//-------------------------------------------------
// This function reads 8 reserved words in the header.
// None of these reserved words are used by the 
// SCI boot loader at this time, they may be used in
// future devices for enhancments.
//-------------------------------------------------
                              
void SCIA_ReservedFn_app()
{
    Uint16 i;
    // Read and discard the 8 reserved words.
    for(i = 1; i <= 8; i++)
    {
       SCIA_FIFO_GetWordData();
    }
    return;
}

//#################################################
// Uint32 SCIA_GetLongData(void)
//-----------------------------------------------
// This routine fetches two words from the SCI-A
// port and puts them together to form a single
// 32-bit value.  It is assumed that the host is
// sending the data in the form MSW:LSW.
//-----------------------------------------------
                                       
Uint32 SCIA_GetLongData_app()
{
    Uint32 longData = (Uint32)0x00000000;

    // Fetch the upper 1/2 of the 32-bit value 
    longData = ( (Uint32)SCIA_FIFO_GetWordData() << 16);
    
    // Fetch the lower 1/2 of the 32-bit value
    longData |= (Uint32)SCIA_FIFO_GetWordData();

    return longData;
}

Uint16 SCIA_GetByteData_app()
{
   Uint16 byteData;
  
   byteData = 0x0000;
   
   // Fetch byte and verify back to the host
   while(SCIARegs.SCIFFRX.bit.RXFIFST == 0) { } 
   byteData =  (Uint16)SCIARegs.SCIRXBUF.bit.RXDT;
   SCIARegs.SCITXBUF = byteData;
   
   return byteData;
}



//#################################################
// Uint16 SCIA_GetWordData(void)
//-----------------------------------------------
// This routine fetches two bytes from the SCI-A
// port and puts them together to form a single
// 16-bit value.  It is assumed that the host is
// sending the data in the order LSB followed by MSB.
//-----------------------------------------------


Uint16 SCIA_GetWordData_app()
{
   Uint16 wordData;
   Uint16 byteData;
  
   wordData = 0x0000;
   byteData = 0x0000;
   
   // Fetch the LSB and verify back to the host
   while(SCIARegs.SCIRXST.bit.RXRDY != 1) { } 
   wordData =  (Uint16)SCIARegs.SCIRXBUF.bit.RXDT;
   SCIARegs.SCITXBUF = wordData;

   // Fetch the MSB and verify back to the host
   while(SCIARegs.SCIRXST.bit.RXRDY != 1) { } 
   byteData =  (Uint16)SCIARegs.SCIRXBUF.bit.RXDT;
   SCIARegs.SCITXBUF = byteData;
   
   // form the wordData from the MSB:LSB
   wordData |= (byteData << 8);

   return wordData;
}

Uint16 SCIA_FIFO_GetWordData()
{
   Uint16 wordData;
   Uint16 byteData;
  
   wordData = 0x0000;
   byteData = 0x0000;
   
   // Fetch the LSB and verify back to the host
   while(SCIARegs.SCIFFRX.bit.RXFIFST < 2) { } 

   wordData =  (Uint16)SCIARegs.SCIRXBUF.bit.RXDT;
//   SCIARegs.SCITXBUF = wordData;

   // Fetch the MSB and verify back to the host
   byteData =  (Uint16)SCIARegs.SCIRXBUF.bit.RXDT;
//   SCIARegs.SCITXBUF = byteData;
   
   // form the wordData from the MSB:LSB
   wordData |= (byteData << 8);

   return wordData;
}

Uint16 SCIA_FIFO_GetWordData_swapEndian()
{
   Uint16 wordData;
   Uint16 byteData;
  
   wordData = 0x0000;
   byteData = 0x0000;
   
   // Fetch the LSB and verify back to the host
   while(SCIARegs.SCIFFRX.bit.RXFIFST < 2) { } 

   byteData =  (Uint16)SCIARegs.SCIRXBUF.bit.RXDT;
//   SCIARegs.SCITXBUF = wordData;

   // Fetch the MSB and verify back to the host
   wordData =  (Uint16)SCIARegs.SCIRXBUF.bit.RXDT;
//   SCIARegs.SCITXBUF = byteData;
   
   // form the wordData from the MSB:LSB
   wordData |= (byteData << 8);

   return wordData;
}




//#################################################
// void SCIA_Init(void)
//-----------------------------------------------------
// This routine initializes the SCI
// 
//-----------------------------------------------------


void SCIA_Init()
{

    // 1 stop bit, No parity, 8-bit character
    // No loopback
    SCIARegs.SCICCR.all = 0x0007;
    
    // Enable TX, RX, Use internal SCICLK
    SCIARegs.SCICTL1.all = 0x0003; 

    // Disable RxErr, Sleep, TX Wake, 
    // Diable Rx Interrupt, Tx Interrupt
    SCIARegs.SCICTL2.all = 0x0000;

    // Relinquish SCI-A from reset
    SCIARegs.SCICTL1.all = 0x0023;
     
    return;
}

//#################################################
// void SCIA_AutobaudLock(void)
//-----------------------------------------------------
// This routine is updates the SCIA baud rate
// using the auto-baud logic
// 
//-----------------------------------------------------



void SCIA_AutobaudLock()
{
    Uint16 byteData;

    // Must prime baud register with >= 1
    SCIARegs.SCIHBAUD = 0;
    SCIARegs.SCILBAUD = 1;
    
    // Prepare for autobaud detection
    // Set the CDC bit to enable autobaud detection
    // and clear the ABD bit      
	SCIARegs.SCIFFCT.bit.ABDCLR = 1;
    SCIARegs.SCIFFCT.all = 0x2000;
    // Wait until we correctly read an 
    // 'A' or 'a' and lock    
    while(SCIARegs.SCIFFCT.bit.ABD != 1) {}

   // After autobaud lock, clear the CDC bit
    SCIARegs.SCIFFCT.bit.CDC = 0;

    while(SCIARegs.SCIRXST.bit.RXRDY != 1) { } 
    byteData = SCIARegs.SCIRXBUF.bit.RXDT;
    SCIARegs.SCITXBUF = byteData;

    return;   
}



void	SCI_SendHexWord(Uint16 temp1)
{
	while(SCIARegs.SCIFFTX.bit.TXFFST>14) ;
	
	SCIARegs.SCITXBUF = temp1>>8;
	SCIARegs.SCITXBUF = temp1;

	return;
}


void	SCI_SendHexLong(Uint32 temp1)
{
	while(SCIARegs.SCIFFTX.bit.TXFFST>12) ;
	
	SCIARegs.SCITXBUF = temp1>>24;
	SCIARegs.SCITXBUF = temp1>>16;
	SCIARegs.SCITXBUF = temp1>>8;
	SCIARegs.SCITXBUF = temp1;

	return;
}







// EOF-------

